#include <stdio.h>
#include <memory.h>
#include <math.h>
#include <sys/time.h>
#include <malloc.h>

#include "tab.h"
#include "stack.h"

/* When compiling ntab make sure BINARY_COUNT and OTHER_COUNTS are both defined in tab.h */

/*
TODO:
1. Blast at level one (should help w/ scheduling problems).
3. Keep clause counts in an array and memcpy instead of pushing on stack (maybe).
4. Optionally keep track of redundant clauses.
5. Make internal-value into a macro (same for walk-clause ?).  Alternatively
could fold internal-value into ur and walk-clause.

BUGS:
1. The -r option (random branch variable selection) overrides the
-h option (branch only on variables occuring in non-horn clauses).
*/

#define TWO_TO_THE_31 2147483648.0 /* 2**31 */
#define MAX_TRIES 100000

/* Variables valued by command line arguments */
char print_assign=0;
int mvar=VARS;

int level;            /* current depth of search tree */
int branch_ends;      /* number of leaves in search tree */
int max_level;        /* Max level in current tree */
int total_height;     /* sum of heights of branches of tree */
int total_levels;     /* total number of nodes in search tree */
int bins_created;     /* total number of binary clauses created since last reset */
int max_var;          /* largest var seen in theory */
int branch_points;    /* number of case splits */

struct stack *all_clauses=NULL; /* text of all clauses */
struct stack *nh_clauses=NULL;  /* all non-horn clauses */
struct stack *ur_stack=NULL;    /* literals to be unit propagated */
struct stack *input_lits=NULL;  /* literals in input */

struct var *var_array;               /* array of information about vars */

struct stack *undo_stack[MAX_LEVEL]; /* array of pointers to stacks of pointers to counters
					decremented in current level */
struct stack *undo;                  /* Always points to current undo stack */
char *val_stack[MAX_LEVEL];          /* stack storing old variable values for
					backtracking */
char *val;                           /* current variable values */
int vars_valued_array[MAX_LEVEL];    /* total number of variables valued at this level */
int *vars_valued;                    /* Invariant: vars_valued = &vars_valued_array[0] */
struct stack *assump_stack;          /* Current assumptions */

#ifdef BINARY_COUNT
/* Binary counts.  These are kept as negative numbers (e.g. -2 means 2 occurances).
   Also, the counts are only valid for unvalued variables. */
char *pbin_stack[MAX_LEVEL];       /* stack storing old binary counts for
					backtracking */
char *pbin;                        /* current binary counts */
char *nbin_stack[MAX_LEVEL];       /* stack storing old binary counts for
					backtracking */
char *nbin;                        /* current binary counts */
#endif

/* Variables set by command line arguments */
char isamp_flg, horn_flg, status_flg, random_flg;
int isamp_max_tries;

/* Dependency directed backtracking */
struct stack **why; /* Array of stacks.  If a variable has a value, its entry
		       here records why it received that value. */
int *mark;
struct stack *nogood;
struct stack *scratch;


/*#define ISAMP_TRACE 1*/


main(argc, argv)
  int argc;
  char *argv[];
{
  int i;

  /* Parse command line arguments */
  isamp_flg=0; /* Default to tableau not isamp */
  horn_flg=0; status_flg=0; random_flg=0;
  while ((--argc > 0) && (*++argv)[0] == '-')
    switch ((*argv)[1]) {
    case 'p':
      print_assign=1;
      break;
    case 'v':
      sscanf(*argv,"-v%d",&mvar);
      break;
    case 'i':
      isamp_flg=1;
      sscanf(*argv,"-i%d",&isamp_max_tries);
      break;
    case 'h':
      horn_flg=1;
      break;
    case 's':
      status_flg=1;
      break;
    case 'r':
      random_flg=1;
      break;
    default:
      printf("ntab: illegal option\n");
      exit(1);
    }

  init_tab();
  input_wff();   /* This inputs the theory and sets max_var to largest var in theory */

  /* Allocate space for recording dependencies */
  why = ((struct stack **) malloc((max_var+1) * sizeof(struct stack *)));
  mark = ((int *) malloc((max_var+1) * sizeof(int)));
  for(i=0;i<=max_var;i++) {why[i] = new_stack(10);}
  nogood = new_stack(10);
  scratch = new_stack(10);

  /* Add in input literals and unit resolve */
  ur_stack = new_stack(max_var);
  while(! empty_stack(input_lits)) /* Unit prop input literals */
    if (!(ur(fpop(input_lits)))) {
#ifdef ARI_COMPILER
      printf("-1 UNSAT\n");
#else
      printf("UNSAT\nContradiction after unit resolution.\n");
#endif
      exit(0);
    }

#ifndef SILENT
  printf("Loaded.  Max var: %d.\n",max_var);
#endif
#ifdef TRACE
  printf("Initial variable values: ");
  show_assign();
#endif

  if (isamp_flg==1) {isamp(); return;}

  if (tab()) {
#ifdef ARI_COMPILER
      show_assign();
#else
      printf("SAT\n");
      if (print_assign) show_assign();
#endif
      print_stats(1);
      if (! model_check()) {fprintf(stderr,"*******MODEL CHECK FAILED********.\n"); exit(1);}
    }
  else {
#ifdef COMPILER
    printf("-1 UNSAT\n");
#else
    printf("UNSAT\n");
#endif
    print_stats(1);
  }
}

print_stats(done)
  int done;
{
  printf("levels %d branch_points %d mean_height %d max_height %d\n",
	 total_levels,
	 branch_points,
	 ((branch_ends>0) ? total_height/branch_ends : 0),
	 max_level);
  if (!done) {
    printf("   [");
    map_stack(assump_stack,printf(" %d",*ptr););
    printf("]\n");
  }
}


init_tab()
{
  int i;

  /* Alloc space for variable size arrays */
  var_array = ((struct var *) malloc(mvar * sizeof(struct var)));
  for(i=0;i<MAX_LEVEL;i++) val_stack[i] = ((char *) malloc(mvar * sizeof(char)));
  val = val_stack[0];
#ifdef BINARY_COUNT
  for(i=0;i<MAX_LEVEL;i++) pbin_stack[i] = ((char *) malloc(mvar * sizeof(char)));
  pbin =pbin_stack[0];
  for(i=0;i<MAX_LEVEL;i++) nbin_stack[i] = ((char *) malloc(mvar * sizeof(char)));
  nbin = nbin_stack[0];
#endif

  init_random();
  level = 0;
  branch_ends = 0;
  total_height = 0;
  max_level = 0;
  total_levels=0;
  bins_created = 0;
  max_var = 0;
  branch_points = 0;

  /* If we run multiple times we really should recollect this storage */
  all_clauses = new_stack(100);
  nh_clauses = new_stack(100);
  /* ur_stack initialized in main routine */
  input_lits = new_stack(10);

  vars_valued = &vars_valued_array[0];
  *vars_valued = 0;

  assump_stack=new_stack(100);

  for(i=1;i<mvar;i++) {
    var_array[i].pos_hits = new_stack(10);
    var_array[i].neg_hits = new_stack(10);
    var_array[i].pos_clauses = new_stack(10);
    var_array[i].neg_clauses = new_stack(10);
    val[i] = 0;
    pbin[i] = 0; nbin[i] = 0;
  }
  for(i=0;i<MAX_LEVEL;i++) {
    undo_stack[i] = new_stack(10);
  }
  undo = undo_stack[0];
  /* We don't init val_stack since its always written into before its read */
}

init_random()
{
  struct timeval tp;
  struct timezone tzp;

  gettimeofday( &tp, &tzp);
  srandom((int) tp.tv_usec);
}


show_assign()
{
  int i;

  /*printf("Assgnment:\n");*/
  printf("\n");
  for(i=1;i<=max_var;i++) if (val[i]>0) printf("%d ",i);
  printf("\n");
}

model_check()
{
  map_stack(all_clauses,
            if (! clause_sat((struct stack *) *ptr)) {
	      struct stack *text = ((struct stack*) *ptr);
	      printf("VIOLATED CLAUSE: (");
	      map_stack(text,printf(" %d",*ptr););
	      printf(")\n");
	      return(0);
	    });
  return(1);
}

clause_sat(s)
struct stack *s;
{
  map_stack(s,if(val[abs(*ptr)] == sign(*ptr)) return(1));
  return(0);
}


/* tab returns 1 iff model found. */
tab()
{
  int var;
  int result= (random_flg ? random_cbl(&var) : cbl(&var));

  if (result == CONTRADICTION) {branch_ends++; total_height += level; return(0);}
  if (result == 0) {            /* model found */
    int i;
    for(i=0;i<max_var;i++) if (val[i]==0) val[i]=-1;
    branch_ends++; total_height += level;
    return(1);
  }

#ifdef DEBUG
  printf("\n");
  for(i=1;i<=max_var;i++)
    printf(":: pbin[%d] = %d, nbin[%d] = %d.\n", i, get_bin(i), i, get_bin(-i));
#endif

#ifdef TRACE
  printf("Guesing %d. ",var);
#endif

  /* Apply the pure literal rule to the branch var only */
  if (! occurs(var)) {
    make_empty(why[abs(var)]);
    value(-var);
    return(tab());
  }
  else if (! occurs(-var)) {
    make_empty(why[abs(var)]);
    value(var);
    return(tab());
  }

  branch_points++;
  if ((status_flg) && ((branch_points & 1023)==1023)) print_stats(0);
  spush(var,assump_stack);
  if (assume(var) && tab()) return(1);
  end_level();
  spop(assump_stack);
  if (!stack_find(nogood,var)) {
#ifdef TRACE
    printf("BBB [level=%d] %d does not occur in current nogood: ",level,var);
    map_stack(nogood,printf(" %d",*ptr););
    printf("\n");
#endif
    return(0);
  }
#ifdef TRACE
  else {
    printf("    [level=%d] %d does occur in current nogood: ",level,var);
    map_stack(nogood,printf(" %d",*ptr););
    printf("\n");
  }
#endif
  stack_delete(nogood,var);
  copy_stack_data(why[abs(var)],nogood);
  spush(-var,assump_stack);
  if (!value(-var)) {
    branch_ends++;
    total_height += level;
    result = 0;
  }
  else result = tab();
  spop(assump_stack);
  return result;
}


isamp()
{
  int i;

  for(i=0;i<isamp_max_tries;i++) {
    if ((status_flg) && ((i & 1023)==1023)) isamp_print_stats(i);
    if (probe()) {
#ifdef ARI_COMPILER
      show_assign();
#else
      printf("SAT\n");
      if (print_assign) show_assign();
#endif
      if (! model_check()) {fprintf(stderr,"*******MODEL CHECK "); exit(1);}
      isamp_print_stats(i);
      return(0);
    }
  }
#ifdef ARI_COMPILER
  printf("-2 GAVE UP\n");
#else
  printf("isamp failed\n");
#endif
  isamp_print_stats(isamp_max_tries);
}

isamp_print_stats(tries)
  int tries;
{
  printf("STATS: tries %d average height %d\n",tries,(tries==0 ? 0 : total_height/tries));
}


probe()
{
  int i, prev_i, result;
  int cur_height=0;

  new_level();
#ifdef DEBUG
  printf("Var values at start of probe: ");
  print_assign();
#endif
  while(1) {
    prev_i=i;
    result=(random_flg ? random_cbl(&i) : cbl(&i));
    if (result == CONTRADICTION) {
      end_level();
      /*if (cur_height==1) {printf("%d\n",-prev_i);value(-prev_i);}*/
#ifdef ISAMP_TRACE
      printf(" LENGTH %d\n",cur_height);
#endif
      return(0);
    }
    if (result == 0) {            /* model found */
      int i;
      for(i=0;i<max_var;i++) if (val[i]==0) val[i]=-1;
#ifdef ISAMP_TRACE
      printf(" LENGTH %d\n",cur_height);
#endif
      return(1);
    }
    
    cur_height++; total_height++;
    if (coin()) i *= -1;
#ifdef TRACE
    printf("Guesing %d. ",i);
#endif
#ifdef ISAMP_TRACE
    /*printf(" %d",i);*/
#endif
    if (! value(i)) {
      end_level();
      /*if (cur_height==1) {printf("%d\n",-i);value(-i);}*/
#ifdef ISAMP_TRACE
      printf(" LENGTH %d\n",cur_height);
#endif
      return(0);
    }
#ifdef TRACE 
    printf("\n");
#endif
  }
}


assume(x)
int x;
{
  int init = bins_created;

  if (!(val[abs(x)] == 0)) {
    fprintf(stderr,"Bogus: assume passed valued variable %d.  Exiting\n",x);
    exit(1);
  }
  new_level();
  make_empty(why[abs(x)]); fpush(x,why[abs(x)]);
  if (value(x)) {
    if ((bins_created - init) == 0) return(1);
    return(bins_created - init);
  }
  else return(0);
}


new_level()
{
  total_levels++;

#ifdef TRACE
  printf("Begin level %d.  Initial vars_valued = %d.\n",level+1,*vars_valued);
#endif

#ifdef LEVEL_CUTOFF
  if (total_levels == LEVEL_CUTOFF) {
#ifdef ARI_COMPILER
    printf("-2 GAVEUP\n");
#else
    printf("LEVEL CUTOFF REACHED\n");
#endif
    print_stats(1);
    exit(0);
  }
#endif

  /* if ((total_levels & 1024) == total_levels) printf("%d stats\n");*/

  if (++level == MAX_LEVEL) {
    fprintf(stderr,"Level exceeded max level.\n");
    exit(1);
  }

  memcpy(val_stack[level],val,(1+max_var)*sizeof(char));   val = val_stack[level];
  memcpy(pbin_stack[level],pbin,(1+max_var)*sizeof(char)); pbin = pbin_stack[level];
  memcpy(nbin_stack[level],nbin,(1+max_var)*sizeof(char)); nbin = nbin_stack[level];

  if (level > max_level) max_level = level;
  make_empty(undo_stack[level]);
  undo = undo_stack[level];
 
  vars_valued_array[level] = *vars_valued;
  vars_valued = &vars_valued_array[level];
}

end_level()
{
  register int *ptr;
  int *top = undo->fill_ptr;

#ifdef TRACE
  printf("End level %d.  Final vars_valued = %d.\n",level,*vars_valued);
#endif

  for(ptr=undo->bottom; ptr < top; ptr++) (* ((int *) *ptr))++;
  level--;
  undo = undo_stack[level];
  val = val_stack[level]; pbin = pbin_stack[level]; nbin = nbin_stack[level];

  vars_valued = &vars_valued_array[level];
}


/* my_rand returns a random integer n: 1 <= n <= max. */
/* We assume that random() returns a number r: 0 <= r < TWO_TO_THE_31. */
my_rand(maxr)
int maxr;
{
  int x;

  x = 1 + floor(maxr * (((double) random())/TWO_TO_THE_31));
  /* printf("my_rand returns %d\n",x); */
  return(x);
}

coin()
{
  return(my_rand(2) - 1);
}
